import copy

from models.cnn import CNN, CNN_IN
from models.fcn import MLP
from models.resnet import resnet18
from models.stargan import Generator
from models.stargan import Discriminator
from torch.autograd import Variable
from torchvision.utils import save_image
import torch
import torch.nn.functional as F
import numpy as np
import os
import time
import datetime
import torchvision

from collections import OrderedDict
import wandb

torch.set_default_dtype(torch.float64)

class Client(object):

    def __init__(self, data_loader, domain, config):
        self.device = torch.device(config.device_name if torch.cuda.is_available() else 'cpu')
        self.loader = data_loader
        self.domain = domain

        # Model configurations.
        self.arch = config.arch
        self.c_dim = config.c_dim
        self.image_size = config.image_size
        self.g_conv_dim = config.g_conv_dim
        self.d_conv_dim = config.d_conv_dim
        self.g_repeat_num = config.g_repeat_num
        self.d_repeat_num = config.d_repeat_num
        self.lambda_cls = config.lambda_cls
        self.lambda_rec = config.lambda_rec
        self.lambda_gp = config.lambda_gp

        # Training configurations.
        self.source_domains = config.source_domains
        self.target_domain = config.target_domain
        self.dataset = config.dataset
        self.batch_size = config.batch_size
        self.num_iters = config.num_iters
        self.num_iters_decay = config.num_iters_decay
        self.g_lr = config.g_lr
        self.d_lr = config.d_lr
        self.n_critic = config.n_critic
        self.beta1 = config.beta1
        self.beta2 = config.beta2
        self.resume_iters = config.resume_iters

        # Test configurations.
        self.use_wandb = config.use_wandb
        self.device = torch.device(config.device_name if torch.cuda.is_available() else 'cpu')
        self.test_iters = config.test_iters

        self.build_model()

    def build_model(self):
        """Create a generator and a discriminator."""
        if self.arch == 'cnn':
            self.D = CNN(1)
        elif self.arch == 'cnn_in':
            self.D = CNN_IN(1)
        elif self.arch == 'fcn':
            self.D = FCN()
        elif self.arch == 'mlp':
            self.D = MLP()
        elif self.arch == 'resnet18':
            self.D = resnet18(1)

        self.optimizer_state = None
        self.D.to(self.device)

    def update_lr(self, d_lr):
        """Decay learning rates of the generator and discriminator."""
        for param_group in self.d_optimizer.param_groups:
            param_group['lr'] = d_lr

    def label2onehot(self, labels, dim):
        """Convert label indices to one-hot vectors."""
        batch_size = labels.size(0)
        out = torch.zeros(batch_size, dim)
        out[np.arange(batch_size), labels.long()] = 1
        return out

    def classification_loss(self, logit, target):
        """Compute binary or softmax cross entropy loss."""
        return F.cross_entropy(logit, target)

    def reset_grad(self):
        """Reset the gradient buffers."""
        self.d_optimizer.zero_grad()

    def log_batch(self, data, label, title):
        grid_img = wandb.Image(
            torchvision.utils.make_grid(
                data[:5], nrow=5,
            ), caption=",".join(map(str, label[:5].numpy().tolist()))
        )
        wandb.log({f'batch_data/{title}': grid_img})

    def test(self, data_loader, log, title):
        correct, total = 0.0, 0.0
        for i, data in enumerate(data_loader):
            x_real = data[0]
            label_org = data[1]

            if self.use_wandb and log and i==0:
                self.log_batch(x_real, label_org, title)

            x_real = x_real.to(self.device)           # Input images.
            label_org = label_org.to(self.device)     # Labels for computing classification loss.

            out_cls = self.D(x_real)
            pred = out_cls.argmax(axis=1)
            correct += (pred == label_org).sum()
            total += pred.shape[0]
        assert total == len(data_loader.dataset)

        return correct/total

    def train(self, i, domain_name):
        """Update the local client for one mini-batch"""

        # # Learning rate cache for decaying.
        d_lr = self.d_lr
        self.d_optimizer = torch.optim.Adam(self.D.parameters(), self.d_lr, [self.beta1, self.beta2])
        if self.optimizer_state != None:
            state = self.d_optimizer.state_dict()
            state['state'] = self.optimizer_state
            self.d_optimizer.load_state_dict(state)
        # self.d_optimizer = torch.optim.SGD(self.D.parameters(), self.d_lr)

        # =================================================================================== #
        #                             1. Preprocess input data                                #
        # =================================================================================== #

        # Fetch real images and labels.
        try:
            x_real, label_org, _ = next(self.data_iter)
        except:
            self.data_iter = iter(self.loader)
            x_real, label_org, _ = next(self.data_iter)

        x_real = x_real.to(self.device)           # Input images.
        label_org = label_org.to(self.device)     # Labels for computing classification loss.

        # =================================================================================== #
        #                             2. Train the discriminator                              #
        # =================================================================================== #

        # Compute loss with real images.
        out_cls = self.D(x_real)
        d_loss_cls = self.classification_loss(out_cls, label_org)

        # Backward and optimize.
        d_loss = d_loss_cls
        self.reset_grad()
        d_loss.backward()
        self.d_optimizer.step()
        self.optimizer_state = self.d_optimizer.state_dict()['state']

        # Logging.
        loss = {}
        with torch.no_grad():
            loss['D/loss_cls'] = d_loss_cls.item()
        loss['D/train_acc'] = self.test(self.loader, i==0, domain_name)
        self.loss = loss


class FedSolver(object):
    """Solver for training and testing StarGAN in Fedrated Learning setting"""

    def __init__(self, loader_dict, domain_idx, test_loader, config):
        """Initialize configurations."""

        # Data loader.
        self.loader_dict = loader_dict
        self.test_loader = test_loader

        self.domain_idx = domain_idx

        # Model configurations.
        self.arch = config.arch
        self.c_dim = config.c_dim
        self.image_size = config.image_size
        self.g_conv_dim = config.g_conv_dim
        self.d_conv_dim = config.d_conv_dim
        self.g_repeat_num = config.g_repeat_num
        self.d_repeat_num = config.d_repeat_num
        self.lambda_cls = config.lambda_cls
        self.lambda_rec = config.lambda_rec
        self.lambda_gp = config.lambda_gp

        # Training configurations.
        self.source_domains = config.source_domains
        self.target_domain = config.target_domain
        self.dataset = config.dataset
        self.batch_size = config.batch_size
        self.num_iters = config.num_iters
        self.num_iters_decay = config.num_iters_decay
        self.g_lr = config.g_lr
        self.d_lr = config.d_lr
        self.n_critic = config.n_critic
        self.beta1 = config.beta1
        self.beta2 = config.beta2
        self.resume_iters = config.resume_iters

        # Test configurations.
        self.test_iters = config.test_iters

        # Miscellaneous.
        self.use_tensorboard = config.use_tensorboard
        self.use_wandb = config.use_wandb
        self.device = torch.device(config.device_name if torch.cuda.is_available() else 'cpu')
        self.run_name = config.run_name

        # Directories.
        self.log_dir = config.log_dir
        self.model_save_dir = config.model_save_dir

        # Step size.
        self.sync_step = config.sync_step
        self.log_step = config.log_step
        self.sample_step = config.sample_step
        self.model_save_step = config.model_save_step
        self.lr_update_step = config.lr_update_step
        self.vis_step = config.vis_step

        self.config = config
        self.vis_batch = next(iter(loader_dict[self.source_domains[2]]))[0]

        # Build the model and tensorboard.
        self.build_model()
        self.init_clients()

        # Setup logger
        if self.use_tensorboard:
            self.build_tensorboard()
        if self.use_wandb:
            self.init_wandb(config)

    def build_model(self):
        """Create a generator and a discriminator."""
        if self.arch == 'cnn':
            self.D = CNN(1)
        elif self.arch == 'cnn_in':
            self.D = CNN_IN(1)
        elif self.arch == 'fcn':
            self.D = FCN()
        elif self.arch == 'mlp':
            self.D = MLP()
        elif self.arch == 'resnet18':
            self.D = resnet18(1)

        self.d_optimizer = torch.optim.Adam(self.D.parameters(), self.d_lr, [self.beta1, self.beta2])
            
        self.D.to(self.device)

    def print_network(self, model, name):
        """Print out the network information."""
        num_params = 0
        for p in model.parameters():
            num_params += p.numel()
        print(model)
        print(name)
        print("The number of parameters: {}".format(num_params))

    def restore_model(self, resume_iters):
        """Restore the trained generator and discriminator."""
        print('Loading the trained models from step {}...'.format(resume_iters))
        D_path = os.path.join(self.model_save_dir, '{}_{}-D.ckpt'.format(self.target_domain, resume_iters))
        self.D.load_state_dict(torch.load(D_path, map_location=lambda storage, loc: storage))

    def build_tensorboard(self):
        """Build a tensorboard logger."""
        from logger import Logger
        self.logger = Logger(self.log_dir)

    def init_wandb(self, args):
        '''Initialize wandb project'''
        wandb.init(project=args.project, entity=args.entity, config=args, name=args.run_name)

    def update_lr(self, d_lr):
        """Decay learning rates of the generator and discriminator."""
        for param_group in self.d_optimizer.param_groups:
            param_group['lr'] = d_lr

    def denorm(self, x):
        """Convert the range from [-1, 1] to [0, 1]."""
        out = (x + 1) / 2
        return out.clamp_(0, 1)

    def label2onehot(self, labels, dim):
        """Convert label indices to one-hot vectors."""
        batch_size = labels.size(0)
        out = torch.zeros(batch_size, dim)
        out[np.arange(batch_size), labels.long()] = 1
        return out

    def classification_loss(self, logit, target):
        """Compute binary or softmax cross entropy loss."""
        return F.cross_entropy(logit, target)

    def average_model(self, coeffs=None):
        """Average the central model from each client """
        if not coeffs:
            coeffs = [1/len(self.source_domains) for _ in range(len(self.source_domains))]

        com_D = Discriminator(self.image_size, self.d_conv_dim, self.c_dim, self.d_repeat_num).to(self.device)
        averaged_D_weights = OrderedDict()
        for i, domain in enumerate(self.source_domains):
            local_D_weight = self.clients_dict[domain].D.state_dict()
            for key in self.D.state_dict().keys():
                if i > 0:
                    averaged_D_weights[key] += coeffs[i] * local_D_weight[key]
                else:
                    averaged_D_weights[key] = coeffs[i] * local_D_weight[key]
        self.D.load_state_dict(averaged_D_weights)

    def transmit_model(self):
        """Send central model to each client"""
        for domain in self.source_domains:
            self.clients_dict[domain].D = copy.deepcopy(self.D)

    def init_clients(self):

        # Create clients
        clients_dict = dict()
        for domain in self.source_domains:
            clients_dict[domain] = Client(self.loader_dict[domain], domain, self.config)
        self.clients_dict = clients_dict

        # synchronize the model
        self.transmit_model()

    def log_batch(self, data, label, title):
        grid_img = wandb.Image(
            torchvision.utils.make_grid(
                data[:5], nrow=5,
            ), caption=",".join(map(str, label[:5].numpy().tolist()))
        )
        wandb.log({f'batch_data/{title}': grid_img})

    def test(self, data_loader, log, title):
        correct, total = 0.0, 0.0
        for i, data in enumerate(data_loader):
            x_real = data[0]
            label_org = data[1]

            if self.use_wandb and log and i==0:
                self.log_batch(x_real, label_org, title)

            x_real = x_real.to(self.device)           # Input images.
            label_org = label_org.to(self.device)     # Labels for computing classification loss.

            out_cls = self.D(x_real)
            pred = out_cls.argmax(axis=1)
            correct += (pred == label_org).sum()
            total += pred.shape[0]
        assert total == len(data_loader.dataset)

        return correct/total

    def train(self):
        """Train StarGAN with FedAvg"""

        # Start training from scratch or resume training.
        start_iters = 0
        if self.resume_iters:
            start_iters = self.resume_iters
            self.restore_model(self.resume_iters)
            self.transmit_model()

        # Start training.
        print('Start training...')
        start_time = time.time()


        for i in range(start_iters, self.num_iters):

            # =================================================================================== #
            # 1. Train local clients for each mini-batch                                          #
            # =================================================================================== #
            for domain in self.source_domains:
                self.clients_dict[domain].train(i, domain)

            # =================================================================================== #
            # 2. Synchronize with central and each local client                                   #
            # =================================================================================== #
            if (i + 1) % self.sync_step == 0:

                # aggregate for central model
                self.average_model()

                # transmit central model to each client
                self.transmit_model()


            # =================================================================================== #
            # 3. Logging                                                                          #
            # =================================================================================== #

            # Print out training information.
            if (i + 1) % self.log_step == 0:

                # first aggregate the loss information from each client
                loss = {}
                coeffs = [1 / len(self.source_domains) for _ in range(len(self.source_domains))]
                # coeffs = [1 for _ in range(len(self.source_domains))]
                for idx, domain in enumerate(self.source_domains):
                    local_loss = self.clients_dict[domain].loss
                    #print(domain, local_loss)
                    for key in local_loss.keys():
                        if idx == 0:
                            loss[key] = coeffs[idx] * local_loss[key]
                        else:
                            loss[key] += coeffs[idx] * local_loss[key]

                et = time.time() - start_time
                et = str(datetime.timedelta(seconds=et))[:-7]
                log = "Elapsed [{}], Iteration [{}/{}]".format(et, i + 1, self.num_iters)
                loss['D/test_acc'] = self.test(self.test_loader, i==9, 'test')
                for tag, value in loss.items():
                    log += ", {}: {:.4f}".format(tag, value)
                print(log)

                if self.use_tensorboard:
                    for tag, value in loss.items():
                        self.logger.scalar_summary(tag, value, i + 1)

                if self.use_wandb:
                    wandb.log(loss)
                    wandb.log({'d_lr': self.d_lr})

            # =================================================================================== #
            # 4. Miscellaneous                                                                    #
            # =================================================================================== #

            # Save model checkpoints.
            if (i + 1) % self.model_save_step == 0:

                D_path = os.path.join(self.model_save_dir, '{}_domain{}_last-D.ckpt'.format(self.dataset,
                                                                                            self.target_domain))
                torch.save(self.D.state_dict(), D_path)
                print('Saved model checkpoints into {}...'.format(self.model_save_dir))

            # Decay learning rates.
            if (i + 1) % self.lr_update_step == 0 and (i + 1) > (self.num_iters - self.num_iters_decay):
                d_lr = self.clients_dict[self.source_domains[0]].d_lr

                d_lr -= (d_lr / float(self.num_iters_decay))
                for domain in self.source_domains:
                    self.clients_dict[domain].update_lr(d_lr)
                self.update_lr(d_lr)
                print('Decayed learning rates, d_lr: {}.'.format(d_lr))
